#!/usr/bin/perl
=begin purpose
  Software: linebreak.pl v1.2

  Purpose: To break long lines in text into shorter according provided length parameters
  Usage format: perl linebreak.pl [-big|-gradually] [file_name] [start_position max_line_length]

  Usage examples:
     linebreak.pl -help
     linebreak.pl -version
     linebreak.pl -big a.txt 70 90
     linebreak.pl -gradually 70 90
     linebreak.pl a.txt 70 90
     linebreak.pl 70 90

  (c)2024, Vartiklis, Jonas Skendelis. The linebreak.pl is distributed under MIT License (see license.txt file) allowing to
	   use, copy, modify, merge, publish, distribute, sublicense, and/or sell its copies. 
  Its copyright notice and license text shall be included in all copies or
  substantial portions of this Software.
=end purpose
=cut

$COPYRIGHT = '(c)2024, Vartiklis, Jonas Skendelis. The linebreak.pl is distributed
under MIT License (see license.txt file) allowing to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell its copies.';

$CURRENT_VERSION = "linebreak.pl v1.2\n\n";
$DEFINITION = "To break long lines in text into shorter (v.1.2, 2024).\n\n";
$SHORT_HELP = "Format: linebreak [-big|-gradually] [file_name] start_position max_line_length\n";
$START_POS_TXT = 'The starting position (%s) ';
$MAX_LENGTH_TXT = 'The maximal line length (%s) ';
$NOT_INTEGER = "is not integer number.\n\n";
$FILE_NOT_EXIST = "The file '%s' does not exist.\n\n";
$FILE_NOT_TEXT = "The file '%s' is not textual.\n\n";
$FILE_NOT_READABLE = "The file '%s' is not readable.\n\n";
$MAX_LENGTH_LOWER = "The maximal length (%d) is lower than start position (%d) of break;\nit was accepted as equal to start position.";
$ERROR_PREFIX = 'Error: ';
$WARNING_PREFIX = 'Warning: ';

$FULL_HELP = 'Formats:
perl linebreak.pl [-big|-gradually] [file_name] start_position max_line_length
perl linebreak.pl -help
perl linebreak.pl -version

Default values of line length are 68 and 80.

Detailed description

Two parameters are used to control splitting: one to define the minimal length
and another to define maximal length. A line part till minimal length is kept
as is and then further the space is searching to make the split in that
position. If space is not found then break happens at maximal length position.

The command below reads a text from standard input (console) and breaks line
leaving at least of 70 symbols but not longer as 90 symbols. The break will
be done at first space after 70th position. The result will be to standard
output stream (console).

perl linebreak.pl 70 90

But it is possible to redirect a result into a file (by using >), for example, 
perl linebreak.pl 70 90 > splited.txt

The input file could be taken from redirection into input stream (by using <),
for example,

perl linebreak.pl 70 90 <text.txt
or transferring in pipe (by using | ), for example,
more text.txt | perl linebreak.pl 70 90
or providing in first position of command, for example,
perl linebreak.pl text.txt 70 90

By default teh entire text is read at once. But if the file is big or computer
resources are low, it would be better to read the file line by line.
That is allowed by parameter -big or -gradually provided in first position.
If this parameter is set, the text will be read line by line. For example:
perl linebreak.pl -big text.txt 70 90
perl linebreak.pl -big 70 90
or
perl linebreak.pl -gradually text.txt 70 90
perl linebreak.pl -gradually 70 90

';

$SHOW_SHORT_HELP = 1;
$SHOW_FULL_HELP = 2;
$SHOW_VERSION = 9;

$ERROR_MSG = '';

$BIG = 0;
$START_POS = 68;
$MAX_LENGTH = 80;

my $posIndex = definePositionIndex(@ARGV);
my $showHelpText = showHelpOrVersion(@ARGV, $posIndex);

if ( $posIndex == -1 || $showHelpText ) {
   if ($showHelpText == $SHOW_FULL_HELP) {
      $ERROR_MSG = $FULL_HELP.$COPYRIGHT;
   } else {
      if ($showHelpText == $SHOW_VERSION) {
         $ERROR_MSG = $CURRENT_VERSION.$COPYRIGHT;
      } else {
         $ERROR_MSG = $SHORT_HELP;
      }
   }
}

my $fileIndex = 0;
if ( $posIndex != 0 ) {
   if ( $#ARGV != -1 ) {
     if (substr($ARGV[0], 0, 1) eq '-') {
        $BIG = 1;
        if ( $#ARGV == 3 ) {
           $fileIndex = 1;
        } else {
           if ( $posIndex == 1 ) {
              if ($#ARGV == 2) {
                  $ERROR_MSG = $SHORT_HELP;
                  $fileIndex = -1;
               } else {
                  $fileIndex = ($posIndex == 1) ? 0 : -1;
               }
           } else {
              $fileIndex = ($posIndex == 12 && $#ARGV == 1) ? 1 : -1;
           }
        }
     } else {
        if ( $posIndex == 1 ) {
           $fileIndex = ($posIndex == 1 && $#ARGV == 2) ? 0 : -1;
        } else {
            $fileIndex = ($posIndex == 12 && $#ARGV == 0) ? 0 : -1;
        }
     }
   } else {
      $fileIndex = ($posIndex == 12) ? 0 : -1; 
   }

   if ($fileIndex != -1 && $posIndex < 11) {
      if (length($ERROR_MSG) == 0 && ! -e $ARGV[$fileIndex]) {
         $ERROR_MSG = $ERROR_PREFIX.sprintf($FILE_NOT_EXIST, $ARGV[$fileIndex]).$SHORT_HELP;
      }
      if (length($ERROR_MSG) == 0 && ! -T $ARGV[$fileIndex]) {
         $ERROR_MSG = $ERROR_PREFIX.sprintf($FILE_NOT_TEXT, $ARGV[$fileIndex]).$SHORT_HELP;
      }
      if (length($ERROR_MSG) == 0 && ! -R $ARGV[$fileIndex]) {
         $ERROR_MSG = $ERROR_PREFIX.sprintf($FILE_NOT_READABLE, $ARGV[$fileIndex]).$SHORT_HELP;
      }
   }
}

if ( $posIndex  < 11 && (!$BIG && $#ARGV == 1)) {
   $ERROR_MSG = $SHORT_HELP; 
}

if ( $posIndex  < 11 && !(!$BIG && $#ARGV == 1) ) {
   if (length($ERROR_MSG) == 0 && ! isInteger($ARGV[$posIndex])) {
       $ERROR_MSG = $ERROR_PREFIX.sprintf($START_POS_TXT, $ARGV[$posIndex]).$NOT_INTEGER.$SHORT_HELP;
   }
   if (length($ERROR_MSG) == 0 && ! isInteger($ARGV[$posIndex+1])) {
       $ERROR_MSG = $ERROR_PREFIX.sprintf($MAX_LENGTH_TXT, $ARGV[$posIndex+1]).$NOT_INTEGER.$SHORT_HELP;
   }
}

if ( length($ERROR_MSG) > 0 ) {
   print STDERR $DEFINITION.$ERROR_MSG;
   exit;
}

if ( $fileIndex == -1) {
   $in = *STDIN;
} else {
   $DATFILE = $ARGV[$fileIndex];

   open (IN, "<$DATFILE");
   $in = *IN;
}

if ( $posIndex < 11 ) { # used not default values
   $START_POS = $ARGV[$posIndex];
   $MAX_LENGTH = $ARGV[$posIndex+1];
}

if (length($ERROR_MSG) == 0 && $START_POS > $MAX_LENGTH) {
   print STDERR $WARNING_PREFIX.sprintf($MAX_LENGTH_LOWER, $MAX_LENGTH, $START_POS)."\n\n";
   $MAX_LENGTH = $START_POS;	
}
        
if ( !$BIG ) {
   @lines = <$in>;
   close $in;

   if ($#lines < 0) {
      exit;
   }
   for ($nr = 0; $nr <= $#lines; $nr++ ) {
       my $broken = &splitline($lines[$nr], $START_POS, $MAX_LENGTH);
       print $broken;
   }

} else {
    while (! eof($in)) {
       my $curline = <$in>;
       my $brokenline = &splitline($curline, $START_POS, $MAX_LENGTH);
       print $brokenline;
    }
    close ($in);
 }

# ------------------- Subroutines area -----------------------------
sub splitline {
   local ($line) = $_[0];
   local ($start_pos) = $_[1];
   local ($threshold) =  $_[2]-$_[1];
   local ($performed);

   while (length($line) > $start_pos && length($line) != 0) {
       $performed .= substr($line, 0, $start_pos);
       $line = substr($line, $start_pos);
       $slice = substr($line, 0, $threshold);
       $breakpos = index($slice, ' ');
       if ($breakpos < 0 ) {
          $performed .= $slice;
          $line = substr($line, $threshold);
       } else {
          $performed .= substr($slice, 0, $breakpos);
          $line = substr($line, $breakpos + 1);
       } 
       $performed.= "\n";
   }
   $performed .= $line;
   return $performed;
}

sub showHelpOrVersion {
  if (@_ == 2 && isHelpArg($_[0])) {
     return $SHOW_FULL_HELP;
  }
  if (@_ == 2 && isVersionArg($_[0])) {
     return $SHOW_VERSION;
  }

  if ($_[@_-1] == -1) {
     return $SHOW_SHORT_HELP;
  }
  if (@_ <= 2) {
     return 0;
  }  
  return ( @_ > 3+$_[@_-1]);
}

sub isHelpArg {
   local ($param) = $_[0];
   return lc($param) eq '-h' || lc($param) eq '-help' || lc($param) eq '--h' || lc($param) eq '--help';
}

sub isVersionArg {
   local ($param) = $_[0];
   return lc($param) eq '-v' || lc($param) eq '-version' || lc($param) eq '--v' || lc($param) eq '--version';
}

sub isInteger {
   return $_[0] =~/^\d+$/;
}

sub definePositionIndex {
   local ($param) = $_[0];

   if (@_ == 1 && isHelpArg($param, 1)) {
         return -1; # case linebreak -help
   }

   if ( @_ > 4 ) {
       return -1;  # case linebreak something
   }
   if ( @_ < 2) {
     return 11+@_;
   }

   if (substr($param, 0, 1) eq '-') {
       if ( lc($param) eq '-big' || lc($param) eq '-gradually') {
          if (@_ == 2) {
              return 12;
          }
	  return ( @_ == 3) ? 1 : 2; 
       } else {
          return -1;
       }
   }
   return 1;
}
